import React, { useEffect, useState } from 'react';
import { Capacitor } from '@capacitor/core';
import {
    PushNotifications,
    Token,
    PushNotificationSchema,
    ActionPerformed,
    RegistrationError,
} from '@capacitor/push-notifications';
import {getToken} from "./front";
import axios from 'axios';
import { useHistory } from 'react-router-dom';

export const usePushNotifications = () => {
    const [token, setToken] = useState<string | null>(null);
    let history = useHistory();
    useEffect(() => {
        console.log(import.meta.env.VITE_BASE_URL + 'user-phone-fcm');
        if (Capacitor.getPlatform() === 'web') {
            console.log('Push notifications not available on web.');
            return;
        }
        // --- Request permission ---
        PushNotifications.requestPermissions().then(result => {
            if (result.receive === 'granted') {
                // --- Register with Apple / Google to receive push via FCM ---
                PushNotifications.register();
            } else {
                PushNotifications.requestPermissions().then((res) => {
                    if (res.receive !== "denied") {
                        PushNotifications.register();
                    }
                });
            }
        });

        // --- On successful registration, receive the FCM token ---
        PushNotifications.addListener('registration', (receivedToken: Token) => {
            console.log('Push registration success, token: ' + receivedToken.value);
            setToken(receivedToken.value);
            axios.post(import.meta.env.VITE_BASE_URL + 'user-phone-fcm', { fcm_token: receivedToken.value }, {
                headers: {
                    Authorization: `Bearer ${getToken}`,
                }
            }).then((res) => {
                console.log(import.meta.env.VITE_BASE_URL + 'user-phone-fcm');

            }).catch(err => {
                // console.log(err);
            });
        });

        // --- Handle registration errors ---
        PushNotifications.addListener('registrationError', (error: RegistrationError) => {
            console.error('Error on registration: ' + JSON.stringify(error));
        });

        // --- Handle incoming notifications when the app is in the foreground ---
        PushNotifications.addListener(
            'pushNotificationReceived',
            (notification: PushNotificationSchema) => {
                console.log('Push received in foreground: ' + JSON.stringify(notification));
                const title = notification.title || 'Notification';
                const body = notification.body || 'You have a new message.';
            }
        );

        // --- Handle notification tap action (when app is opened from notification) ---
        PushNotifications.addListener(
            'pushNotificationActionPerformed',
            (notificationAction: ActionPerformed) => {
                let actionsUrl = notificationAction.notification.data.actions,
                    newUrl = actionsUrl.includes("/ar/")
                        ? actionsUrl.replace("/ar/", "/")
                        : actionsUrl.includes("/en/")
                            ? actionsUrl.replace("/en/", "/")
                            : actionsUrl,
                    url = newUrl.includes("https://jdr.it-rays.net/")
                        ? newUrl.replace("https://jdr.it-rays.net/", "/")
                        : newUrl;
                if (url.includes("?id=")) {
                    url = url.split("?id=").shift();
                }

                if (url.includes("order-details")) {
                    url = import.meta.env.VITE_BASE_URL + url;
                } else if (url.includes("management-messages")) {
                    url = import.meta.env.VITE_BASE_URL + "/services" + url;
                } else if (url.includes("complaint")) {
                    url = import.meta.env.VITE_BASE_URL + "/orders" + url;
                } else if (
                    url.includes("profile/edit") ||
                    url.includes("seeker/edit")
                ) {
                    url = import.meta.env.VITE_BASE_URL + "/settings/profile";
                } else if (url.includes("contact-us")) {
                    url = import.meta.env.VITE_BASE_URL + "/settings/contactus";
                } else if (url.includes("terms")) {
                    url = import.meta.env.VITE_BASE_URL + "/settings/terms";
                } else if (url.includes("privacy-policy")) {
                    url = import.meta.env.VITE_BASE_URL + "/settings/privacy";
                } else if (url.includes("about-us")) {
                    url = import.meta.env.VITE_BASE_URL + "/settings/about";
                } else {
                    url = import.meta.env.VITE_BASE_URL;
                }
                history.push(url);

            }
        );

        return () => {
            console.log('Removing push notification listeners');
            PushNotifications.removeAllListeners().catch(err =>
                console.error("Error removing listeners", err)
            );
        };
    }, []);

    // // You can return the token or other state if needed by components
    // return { fcmToken: token };
};


