import axios from "axios";
import React, { createContext, useContext, useState, useEffect, useRef } from "react";
import useSocketIOStore from "../contexts/socketIO";
import { getUserData } from "../utils/front";
import CallScreen from "../components/communication/CallScreen";

interface User {
  id: string;
  email: string;
  name?: string;
}

interface AuthContextType {
  user: User | null;
  isAuthenticated: boolean;
  isLoading: boolean;
  error: string;
  showDialog: (component: any) => void;
  hideDialog: (component: any) => void;
  login: (email: string, password: string) => Promise<void>;
  logout: () => void;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState("");
  const [dialog, setDialog] = useState<React.ReactNode | null>(null);
  const socket = useSocketIOStore((state:any) => state.socket);
  const msgsTikRef = useRef(null);
  const currentUser = getUserData();

  useEffect(() => {
    const storedUser = localStorage.getItem('hub-user-session');
    if (storedUser) {
      setUser(JSON.parse(storedUser));
    }
    setIsLoading(false);
    setError("");
  }, []);

  useEffect(() => {
    let currentSocket = socket;    
    if (currentSocket) {
      const handleHaveACall = ({ callerId, caller, orderChat, hasVideo }: { callerId: string; caller: number; orderChat: { id: string }; hasVideo: boolean }) => {        
        // history.push({
        //   pathname: `/call`,
        //   state: { info: {
        //       orderChat: orderChat,
        //       hasVideo: hasVideo,
        //       ringType: 'recieving'
        //     }
        //   }
        // });
        showDialog(
          <CallScreen callType="recieving" orderChat={orderChat} hasVideo={hasVideo} />
        );
      };

      const handleHaveAMessage = ({chatId, newMessage, type, chat, action, sendNotification}: { chatId: string; newMessage: { senderId: string }; type: string; chat: any; action: any; sendNotification: any }) => {        
        if (msgsTikRef.current && newMessage.senderId != currentUser.id) {
          msgsTikRef.current.play();
        }
      };

      currentSocket.on('have-a-call', handleHaveACall);
      currentSocket.on('have-a-message', handleHaveAMessage);
    } else {
      // console.log('Socket not connected');
    }
  }, [socket]);

  const login = async (email: string, password: string) => {
    setIsLoading(true);
    setError("");
    try {
      await new Promise((resolve, reject) => {
        // setTimeout(() => {
          axios
            .post(import.meta.env.VITE_BASE_URL + "login", {
              email: email,
              password: password,
            })
            .then((res) => {
              const data = res.data;
              if (data.status === 'success') {
                if (password.length < 6) {
                  reject(new Error("Invalid credentials"));
                  return;
                }
                setUser(data.user);
                localStorage.setItem('hub-token', data.token);
                localStorage.setItem('hub-user-session', JSON.stringify(data.user));
                resolve(data);
              } else if (data.status === 0) {
                setIsLoading(false);
                setError(data.message);
                resolve(data);
              }
            })
            .catch((err) => {
              setIsLoading(false);
              setError(err);
              reject(err);
            });
        // }, 100);
      });
    } catch (error) {
      setIsLoading(false);
      setError(error.message || error);
    } finally {
      setIsLoading(false);
    }
  };

  const logout = () => {
    setUser(null);
    localStorage.removeItem('hub-token');
    localStorage.removeItem('hub-user-session');
  };

  const showDialog = (component: React.ReactNode) => {
    setDialog(component);
  };
  
  const hideDialog = () => {
    setDialog(null);
  };

  return (
    <AuthContext.Provider
      value={{
        user,
        isAuthenticated: !!user,
        isLoading,
        error,
        showDialog,
        hideDialog,
        login,
        logout,
      }}
    >
      {children}
      <audio ref={msgsTikRef} src="/sounds/new-message.mp3" preload="auto"></audio>
      { dialog }
    </AuthContext.Provider>
  );
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider");
  }
  return context;
}
