"use client"
import {useEffect, useRef, useState} from 'react';
import styles from './LiveChat.module.css';
import {useParams, useHistory} from 'react-router-dom';
import MessageType from './MessageType';
import {getToken, getUserData} from "../../utils/front";
import {FILEFORMATPATH} from "../../utils/common";
import axios from "axios";
import WaveSurfer from 'wavesurfer.js';
import RecordPlugin from 'wavesurfer.js/dist/plugins/record.esm.js';
import Modal from "../Modal";
import {getPermissions} from "./utils/methods";
import MediaStatus from "./MediaStatus";
import { Keyboard } from '@capacitor/keyboard';

function ChatForm({socket, orderChat, setMessages, replyMsg, setReplyMsg, setOrderChat, type, chatUser, setChatPageId, editMsg, setEditMsg, updateMessageSeenState}
:{ socket:any, orderChat:any, setMessages:any, replyMsg:any, setReplyMsg:any, setOrderChat:any, type:any, chatUser:any, setChatPageId:any, editMsg:any, setEditMsg:any, updateMessageSeenState:any}
) {
  const messageInput = useRef<any>(null);
  const [file, setFile] = useState<any>(null);
  const [fileError, setFileError] = useState<any>(null);
  const [isSending, setIsSending] = useState<number>(0);
  const { id } = useParams();
  const user = getUserData();
  const user_token = getToken();
  const fileInput = useRef<any>(null);
  const acceptedExt = ['.png', '.jpeg', '.jpg', '.pdf', '.doc', '.docx', '.xls', '.xlsx', '.docs', '.txt', '.gif', '.ppt', '.pptx', '.rtf', '.zip', '.rar', '.svg'];
  const history = useHistory();
  const formRef = useRef(null);
  const [audioBlob, setAudioBlob] = useState(null);
  const modal =useRef<any>(null);
  const [errorStatus, setErrorStatus] = useState<any>('');
  const msgTikRef = useRef(null);
  const waveformRef = useRef<any>(null);
  const wavesurferRef = useRef<any>(null);
  const recordPluginRef = useRef<any>(null);
  const [isRecording, setIsRecording] = useState(false);
  const [isPlaying, setIsPlaying] = useState(false);
  const [inAudioRecord, setInAudioRecord] = useState(false);
  const [audioUrl, setAudioUrl] = useState<any>(null);
  const [recordDuration, setRecordDuration] = useState<any>(0);
  const [isPaused, setIsPaused] = useState(false);
  const [typing, setTyping] = useState(false);

  useEffect(() => {
    if (isSending === 1){
      sendMessageOrFile(false);
    }
  }, [socket]);

  useEffect(() => {
    if (isRecording){
      modal.current?.dismiss();
    }
  }, [isRecording]);

  function sendMessage() {
    if (file || messageInput.current?.value || audioBlob) {
      setIsSending((prev) => prev += 1);
      setFileError(null);
    } else {
      messageInput.current?.focus();
    }
  }

  useEffect(() => {
    if(isSending === 1){
      if(Object.keys(orderChat).length){
        sendMessageOrFile()
      }else{
        axios.post( import.meta.env.VITE_BASE_URL +'create-chat-wth-user',{
          other_user: chatUser.id
        },{
          headers: { Authorization: `Bearer ${user_token}` }
        }).then(res=>{
          const data = res.data;
          if(data.status === 'redirect'){
            history.replace('/dashboard/chat/' + data.chat_id);
          }else if(data.status == 0){
            history.replace('/dashboard/communicator');
          }else if(data.status === 'success'){
            setOrderChat(data.chat);
            setChatPageId(data.chat.id);
          }
        })
      }
    }
  }, [isSending]);

  function sendMessageOrFile(redirect = false){
    if (isSending === 1) {
      if(audioBlob){
        uploadAudioFile();
      }else if(file){
        uploadFile();
      }else{
        if (messageInput.current.value) {
          socketSendMessage({
            message: messageInput.current.value,
            senderId: user['id'],
            messageType: 'text',
            chatId: orderChat.id,
            redirectPage:redirect,
          });
        }
      }
    }
  }

  useEffect(() => {
    if(editMsg){
      messageInput.current.value = editMsg.message;
    }
  }, [editMsg]);

  function socketSendMessage(messageObject:any) {
    if (replyMsg) {
      messageObject.replyOn = replyMsg._id;
    }

    if(editMsg){
      messageObject.edited = true;
      messageObject.msgId = editMsg._id;
    }
    messageObject.users = orderChat.users;
    messageObject.type = type;
    messageObject.chat = orderChat;
    socket?.emit('send-message', messageObject, ({ newMessage, chatId, actionType }: { newMessage:any, chatId:any, actionType:any }) => {
        if (chatId == orderChat.id) {
          if(actionType === 'edit'){
            updateMessageSeenState([newMessage]);
          }else{
            setMessages((prevMessages:[]) => [...prevMessages, newMessage]);
          }
          if(messageInput.current){
            messageInput.current.value = '';
            setTyping(false);
          }

          if(messageObject.redirectPage){
            history.replace('/dashboard/chat/' + chatId);
          }
          socket?.emit('typing', { chatId: id, typing: false, userId: user['id'], type, users: orderChat.users });
          if (file) {
            setFile(null);
          }
          if(audioBlob){
            setAudioBlob(null);
            stopRecording();
          }
          setIsSending(0);
          if (msgTikRef.current) {
            msgTikRef.current.play();
          }      
          if (replyMsg) {
            setReplyMsg(null);
          }
          if(editMsg){
            setEditMsg(null);
          }
        }
      });
  }

  function uploadAudioFile() {
    let formData = new FormData();
    if (audioBlob) {
      formData.append('audio', audioBlob, (orderChat.id + '-recording.wav'));
      axios.post(import.meta.env.VITE_BASE_URL + 'upload-audio-file',
          formData,
          { headers: { Authorization: `Bearer ${user_token}` } }
      ).then((res) => {
        const data = res.data;
        if (data.status === 'success') {
          const uploadedFile = data;
          socketSendMessage({
            message: uploadedFile.file_name,
            senderId: user['id'],
            messageType: 'audio',
            chatId: orderChat.id,
          });
        }

      }).catch(err => {
        console.log(err);
      });
    }
  }

  function uploadFile() {
    let formData = new FormData();
    if (file) {
      formData.append('file', file);
      axios.post(import.meta.env.VITE_BASE_URL + 'upload-file',
        formData,
        { headers: { Authorization: `Bearer ${user_token}` } }
      ).then((res) => {
        const data = res.data;
        if (data.status === 'success') {
          const uploadedFile = data;
          const validImageTypes = ['image/jpg', 'image/jpeg', 'image/png'];
          const type = validImageTypes.includes(file.type) ? 'image' : 'document';
          socketSendMessage({
            message: uploadedFile.file_name,
            senderId: user['id'],
            messageType: type,
            chatId: orderChat.id,
          });
        }

      }).catch(err => {
        console.log(err);
      });
    }
  }

  function emitTyping(event:any) {
    if( event.target.value ){
      setTyping(true);
    } else {
      setTyping(false);
    }
    socket?.emit('typing', { chatId:orderChat.id, typing: !!event.target.value, userId: user['id'], type, users: orderChat.users });
  }

  function ShowFileContent() {
    const validImageTypes = ['image/jpg', 'image/jpeg', 'image/png'];
    const typeImage = validImageTypes.includes(file.type);
    const ext = file.name.split('.').pop();
    const size = file.size;
    const sizeWKB = (size / 1024);

    if (typeImage) {
      return <img width={88} height={100} alt="" src={URL.createObjectURL(file)} />
    } else {
      return (
        <div>
          <img alt="document" src={`${FILEFORMATPATH}${ext}.svg`} width={80} height={80} />
          <h6>Document</h6>
          <span>{(sizeWKB < 1000) ? sizeWKB.toFixed(2) : (sizeWKB / 1000).toFixed(2)} {(sizeWKB < 1000) ? 'K.B' : 'M.B'}</span>
        </div>
      )
    }
  }

  function checkFileType() {
    if (fileInput.current?.files.length) {
      const ext = fileInput.current?.files[0].name.split('.').pop();
      const size = fileInput.current?.files[0].size;

      if (acceptedExt.includes('.' + ext)) {
        if ((size / 1024) <= 5000) {
          setFile(fileInput.current?.files[0]);
        } else {
          setFileError('File size should be less than 5MB.');
        }
      } else {
        setFileError('Uploaded files extension should be: ' + acceptedExt.join(', '));
      }
    }
  }

  function messageSender(senderId:any) {
    return orderChat['users']?.find((user:any) => user.id == senderId);
  }

  function textareaKeyPress(event:any) {
      if (event.keyCode == 13 && event.shiftKey == false) {
        sendMessage();
      }
  }

  const updateProgress = (time: number) => {
    return [
      Math.floor((time % 3600000) / 60000),
      Math.floor((time % 60000) / 1000),
    ].map((v) => (v < 10 ? '0' + v : v)).join(':');
  }

  useEffect(() => {

    wavesurferRef.current = WaveSurfer.create({
      container: waveformRef.current,
      waveColor: '#a7a7a7',
      progressColor: '#037b67',
      cursorColor: '#037b67',
      cursorWidth: 13,
      barWidth: 2,
      barHeight: 15,
      barGap: 2,
      height: 15,
      // responsive: true,
      normalize: true,
    });

    recordPluginRef.current = wavesurferRef.current.registerPlugin(
      RecordPlugin.create({
        renderRecordedAudio: true,
        scrollingWaveform: true,
      })
    );

    recordPluginRef.current.on('record-start', () => {
      setInAudioRecord(true);
    });

    recordPluginRef.current.on('record-progress', (time:any) => {
      setRecordDuration(updateProgress(time))
    });

    recordPluginRef.current.on('record-pause', (blob:any) => {
      setAudioBlob(blob);
      const url = URL.createObjectURL(blob);
      setAudioUrl(url);
    });

    recordPluginRef.current.on('record-stop', (blob:any) => {
      setAudioBlob(blob);
      setAudioUrl(null);
      setIsRecording(false);
    });

    recordPluginRef.current.on('play', () => {
      setIsPlaying(true);
    });

    recordPluginRef.current.on('pause', () => {
      setIsPlaying(false);
    });

    recordPluginRef.current.on('finish', () => {
      setIsPlaying(false);
    });

    return () => {
      if (wavesurferRef.current) {
        wavesurferRef.current.destroy();
      }
    };
  }, []);

  const toggleRecording = () => {
    if (isRecording) {
      recordPluginRef.current.stopRecording();
      //setIsRecording(false);
    } else {
      checkMediaPermissions();
    }
  };

  const stopRecording = () => {
    if (isRecording) {
      recordPluginRef.current.stopRecording();
      setIsRecording(false);
    }
  };

  const togglePlayback = () => {
    if (audioUrl) {
      if (isPlaying) {
        wavesurferRef.current.pause();
      } else {
        wavesurferRef.current.play();
      }
    }
  };

  const togglePauseResume = () => {
    if (isPaused) {
      recordPluginRef.current.resumeRecording();
      setIsPaused(false);
    } else {
      recordPluginRef.current.pauseRecording();
      setIsPaused(true);
    }
  };

  async function checkMediaPermissions() {
    try {
      const microphonePermission = await navigator.permissions.query({ name: 'microphone' });
      if(microphonePermission.state == 'granted'){
        recordPluginRef.current.startRecording();
        setIsRecording(true);
      }else if(microphonePermission.state == 'prompt'){
        modal.current?.open();
        setErrorStatus('prompt');
        getPermissions(false).then((stream:any)=>{
          stream.getTracks().forEach((track:any) => track.stop());
          recordPluginRef.current.startRecording();
          setIsRecording(true);
        }).catch(err=>{
          checkMediaPermissions();
          console.log(err);
        });

      }else if(microphonePermission.state == 'denied'){
        modal.current?.open();
        let mediaType = 'Microphone';
        setErrorStatus('denied');
      }

      return {
        microphone: microphonePermission.state === 'granted'
      };
    } catch (error) {
      console.warn('Permissions API not supported, falling back to MediaDevices', error);
      return false;
    }
  }

  useEffect(() => {
    // Set keyboard to resize the app when it appears
    Keyboard.setResizeMode({ mode: 'native' });
    
    // This ensures the plugin is properly cleaned up
    return () => {
      // No need to remove listeners since we're not adding any
    };
  }, []);

  return (
    <>
      <Modal ref={modal}>
        <MediaStatus status={errorStatus} />
      </Modal>
      {fileError && <span className={`ryb-validation-txt ryb-show-error ${styles.errorFiles}`}>{fileError}</span>}
      {replyMsg &&
        <div className={styles.relyMsgWrap}>
          <h6>{messageSender(replyMsg.senderId)['fullName']}</h6>
          <MessageType msg={replyMsg} userId={user.id} messages={[]} />
          <button type='button' onClick={() => setReplyMsg(null)}><i className="ri-close-circle-fill"></i></button>
        </div>
      }
      <div className={`ryb-msg-form ${styles.sendMessageInput}`} ref={formRef}>
        <div className={styles.msgInputTools}>

          <div className={styles.chatMessageTools}>
            {!isRecording &&
              <>
              <i className={`ri-add-line ${styles.msgToolsIcon}`} onClick={() => {fileInput.current.value = ''; fileInput.current?.click()}}></i>
              <input type="file" name="file" ref={fileInput} className="hidden" onChange={checkFileType} accept={acceptedExt.join(', ')} />
              </>
            }
          </div>
          {file &&
            <div className={styles.uploadedFile}>
              <ShowFileContent />
              <span onClick={() => setFile(null)} className={styles.deleteUploaded}><i className="ri-delete-bin-line"></i></span>
            </div>
          }

          <div className={styles.inputMsgWrap}>
            {!isRecording && <textarea ref={messageInput} onKeyDown={(event)=> textareaKeyPress(event)} onChange={emitTyping}></textarea>}
            <div className={`${styles.recordingWrap} ${isRecording ? styles.showRecording : ''}`}>
              <button onClick={togglePauseResume} className={styles.pauseResume}><i className={`ri-${isPaused ? 'mic' : 'pause'}-fill`}></i></button>
              {isPaused && <button onClick={togglePlayback}>
                <i className={`${styles.playRecording} ${isPlaying ? 'ri-pause-mini-fill' : 'ri-play-mini-fill'}`}></i>
              </button>}
              <div ref={waveformRef} className="record-msg-bar"></div>
              <span className={styles.recordDuration}>{recordDuration}</span>
              {!isPaused && <i className={`ri-circle-fill ${styles.recordCircle}`}></i>}
              <button onClick={stopRecording} className={styles.deleteRecording}><i className="ri-delete-bin-5-line"></i></button>
            </div>
          </div>
          
        </div>
        
        { (typing || isRecording || file) &&
          <button className={styles.sendMsg} onClick={sendMessage}>
            {isSending === 1 ? <span className={`ryb-loader-spin ${styles.loadingBtn}`}></span> : <i className={`ri-arrow-up-circle-fill ${styles.sendIcon}`}></i>}
          </button>
        }
        
        { !typing && !isRecording && !file && 
          <div className={styles.voiceMsg}>
            {!isRecording && !typing && 
              <button onClick={toggleRecording} className={styles.MicIcon + ' ' + styles.sendMsg}>
                <i className={isRecording ? 'ri-stop-circle-fill' : 'ri-mic-line'}></i>
              </button>
            }
          </div>
        }

        <audio ref={msgTikRef} src="/sounds/tik.mp3" preload="auto"></audio>

      </div>
      {/* <div className={styles.chatEmojis}>
        {<EmojiPicker open={openEmoji} searchDisabled={true} onEmojiClick={emojiClick} autoFocusSearch={false} lazyLoadEmojis={true} />}
      </div> */}
    </>
  )
}

export default ChatForm;