import React, {useEffect, useState, useRef} from "react";
import {Peer} from "peerjs";
import {getPermissions, stopStream, configration} from './utils/methods';
import {getUserData} from "../../utils/front";
import styles from './CallScreen.module.css';
import {useAuth} from '../../contexts/AuthContext';
import {useLocation} from "react-router-dom";
import useSocketIOStore from "../../contexts/socketIO";
import UserImage from "../UserImage";
import AudioCallBtns from "./AudioCallBtns";
import VideoCallBtns from "./VideoCallBtns";
import Draggable from 'react-draggable';

interface CallScreenProps {
    callType: string;
    orderChat: any;
    hasVideo: boolean;
}

const CallScreen: React.FC<CallScreenProps> = ({ callType, orderChat, hasVideo }) => {

    const { hideDialog } = useAuth();
    const socket = useSocketIOStore((state:any) => state.socket);
    const searchParams = new URLSearchParams(useLocation().search);
    const type = searchParams.get('type');
    const [showCall, setShowCall] = useState(callType);
    const [minimized, setMinimized] = useState(false);
    const showCallRef = useRef<string>('');
    const remoteUserRef = useRef<any>(null);
    const currentUserRef = useRef<any>(null);
    const peerInstance = useRef<any>(null);
    const remotePeerIdRef = useRef<any>(null);
    const peerId = useRef<any>(null);
    const callInstance = useRef<any>(null);
    const localStream = useRef<any>(null);
    const ringToneRef = useRef<any>(null);
    const isVideoCall = useRef(false);
    const ringingInterval = useRef<any>(null);
    const resendCallIntervalRef = useRef(null);
    const minutesRef = useRef(0);
    const secondsRef = useRef(0);
    const user = getUserData();
    const otherUser = orderChat && socket ? orderChat.users.find((usr:any) => usr.id != user.id) : null;
    const nodeRef = useRef(null);
    const currentUserVidRef = useRef(null);
  
    const calculatePositionFromBottom = () => {
        const initialX = 0;
        const bottomPosition = window.innerHeight - 126;
        return { x: initialX, y: bottomPosition };
    };

    const [position, setPosition] = useState(calculatePositionFromBottom);
    const [bounds, setBounds] = useState({
        left: 0, 
        top: 0, 
        right: window.innerWidth - 350,
        bottom: window.innerHeight - 126
    });

    const handleDrag = (e: React.MouseEvent, data: { x: number; y: number }) => {
        // setPosition({ x: data.x, y: data.y });
    };
    
    useEffect(() => {
        const handleResize = () => {
            setPosition(prev => ({
              x: prev.x,
              y: window.innerHeight - 126
            }));
        };
        window.addEventListener('resize', handleResize);
        return () => window.removeEventListener('resize', handleResize);
    }, []);

    const handleDragStop = (e: React.MouseEvent, data: { x: number; y: number }) => {
        setPosition({ x: data.x, y: data.y });
    };

    useEffect(() => {
        
        socket?.on('have-a-call', ({ callerId, orderChat, hasVideo }: { callerId: string; orderChat: any; hasVideo: boolean }) => {
            remotePeerIdRef.current = callerId;
        });

        socket.on('callRecieved', ({ order, recieved }: { order: any; recieved: any }) => {
            clearInterval(resendCallIntervalRef.current);
            resendCallIntervalRef.current = null;
          })

        socket?.on('callOpened', ({ callerId, orderChat, hasVideo }: { callerId: string; orderChat: any; hasVideo: boolean }) => {
            setShowCall('answer');
            clearTimeout(ringingInterval.current);
            clearInterval(resendCallIntervalRef.current);
            ringingInterval.current = null;
            resendCallIntervalRef.current = null;
        });

        socket?.on('user-end-call', ({ orderId }:{orderId:any}) => {
            endCall();
        });

        window.addEventListener('beforeunload', function (event) {
            if (peerInstance.current) {
                endCall(true);
            }
        });

    }, [socket]);

    function callMsg(callType:any, messageType = 'call', callDuration = null) {
        const messageObject = {
            message: callType,
            senderId: user.id,
            messageType,
            chatId: orderChat.id,
            type: type,
            users: orderChat.users,
            callDuration
        }
        // socket.emit('send-message', messageObject, ({ newMessage, chatId }:{ newMessage:any, chatId:any }) => {
        //     if (chatId == orderChat.id) {
        //         setMessages((prevMessages:[]) => [...prevMessages, newMessage]);
        //     }
        // });
    }

    useEffect(() => {
        checkMediaPermissions(hasVideo);
    }, []);

    async function checkMediaPermissions(hasVideo: boolean) {
        try {
            const cameraPermission = await navigator.permissions.query({ name: 'camera' });
            const microphonePermission = await navigator.permissions.query({ name: 'microphone' });
            
            if (cameraPermission.state == 'granted' && microphonePermission.state == 'granted') {
                startCall(hasVideo);
            } else if (cameraPermission.state == 'prompt' || microphonePermission.state == 'prompt') {
                startCall(hasVideo);
            }

            return {
                camera: cameraPermission.state === 'granted',
                microphone: microphonePermission.state === 'granted'
            };
        } catch (error) {
          console.warn('Permissions API not supported, falling back to MediaDevices', error);
          return false;
          // return checkMediaPermissionsFallback();
        }
    }

    function startCall(hasVideo = true) {
        ringingInterval.current = setTimeout(function () {
            let i = 1;
            setShowCall(prev => {
                if (showCall === 'answer') {
                    clearTimeout(ringingInterval.current);
                } else if (prev === 'ringing' || prev === undefined) {
                    if (i) {
                        i = 0;
                        endCall(true, true);
                    }
                }
                return prev;
            });
        }, 60000);
        let peer = new Peer(configration);
        peerInstance.current = peer;
        isVideoCall.current = hasVideo;
        peerInstance.current.on('open', function (id:any) {
            peerId.current = id;
            getPermissions(hasVideo).then((stream) => {
                localStream.current = stream;
                if( localStream.current ){
                    currentUserRef.current.srcObject = stream;
                    currentUserRef.current.play();
                }
                if (ringToneRef.current) {
                    ringToneRef.current.play();
                }
                sendCallSocket(hasVideo);
            }).catch(err => {
                if (peerInstance.current) {
                    peerInstance.current.destroy();
                }
                endCall(true, true);
            });
        });

        peerInstance.current.on('call', function (call:any) {
            callInstance.current = call;
            call.answer(localStream.current);
            call.on('stream', function (stream: MediaStream) {
                if(remoteUserRef.current){
                    remoteUserRef.current.srcObject = stream;
                    remoteUserRef.current.play();
                }
            });
        });

        peerInstance.current.on('disconnected', function () {
            console.log('Disconnected! Attempting to reconnect...');
            endCall(true, true);
        });
        peerInstance.current.on('error', function () {
            console.log('Error peer connection...');
            endCall(true, true);
        });

        if( !callType ){
            setShowCall('ringing');
        }
        
    }

    function openCall() {
        let peer = new Peer(configration);
        peer.on("open", function (id) {
            setShowCall("answer");
            peerInstance.current = peer;
            socket.emit('acceptCall', {
                receiverId: peerId.current,
                order: orderChat.id,
                userId: otherUser.id,
                type: type,
            }, ({userId, orderChat}:{userId:any, orderChat:any})=>{});

            getPermissions(hasVideo).then( (mediaStream) => {

                localStream.current = mediaStream;
                currentUserRef.current.srcObject = mediaStream;
                currentUserRef.current.play();
                
                const call = peer.call(remotePeerIdRef.current, mediaStream);
                callInstance.current = call;
                call.on("stream", function (stream) {
                    remoteUserRef.current.srcObject = stream;
                    remoteUserRef.current.play();
                });

                call.on("close", () => {
                    endCall(true);
                });

                call.on("error", function (src) {
                    endCall(true);
                });
                
            })
            .catch((err) => {
                endCall(true);
            });
        });

        peer.on("disconnected", function (data) {
            console.log(data);
            peer.reconnect();
        });
    }

    function sendCallSocket(hasVideo:any){
        const callContent = {
            callerId: peerId.current,
            caller: 3,
            order: orderChat.id,
            hasVideo: hasVideo,
            orderChat,
            userId: otherUser.id,
            // type: type,
        };
        socket.emit('startCall', callContent, ({userId, orderChat}:{userId:any, orderChat:any})=>{
            // if (!resendCallIntervalRef.current) {
            //     resendCallIntervalRef.current = setInterval(() => {
            //       socket.emit('startCall', callContent);
            //     }, 7000);
            // }
        });
    }

    const stopAllTracks = (stream: MediaStream) => {
        if (!stream) return;
        
        let tracks = stream.getTracks();
        tracks.forEach(track => {
          track.enabled = false;
          track.stop();
        });
        
        const audioTracks = stream.getAudioTracks();
        const videoTracks = stream.getVideoTracks();
        
        audioTracks.forEach(track => {
          track.enabled = false;
          track.stop();
        });
        
        videoTracks.forEach(track => {
          track.enabled = false;
          track.stop();
        });
    };

    function endCall(isEndedByMe = false, isRing = false) {

        if ((isRing || showCall === 'ringing' || showCall == undefined) && isEndedByMe) {
            callMsg(isVideoCall.current ? 'video' : 'audio', 'missedCall');
        }
        if(showCallRef.current && (showCallRef.current !== 'ringing' || showCall != undefined)){
            let secs:any = ((minutesRef.current * 60) + secondsRef.current).toString();
            callMsg(isVideoCall.current ? 'video' : 'audio', 'call', secs);
        }
        minutesRef.current = 0;
        secondsRef.current = 0;

        callInstance.current?.close();
        stopStream(currentUserRef.current);
        stopStream(remoteUserRef.current);

        if (resendCallIntervalRef.current) {
            clearInterval(resendCallIntervalRef.current);
            resendCallIntervalRef.current = null;
        }

        if (ringingInterval.current) {
            clearTimeout(ringingInterval.current);
            ringingInterval.current = null;
        }

        callInstance.current = null;
        isVideoCall.current = false;
        peerInstance.current = null;
        peerId.current = null;

        stopAllTracks(localStream.current);

        // if (localStream.current) {
        //     const tracks = localStream.current.getTracks();
        //     tracks.forEach((track:any) => {
        //         track.enabled = false;
        //         track.stop();
        //         console.log(track);
        //     });
        //     // localStream.current = null;
        // }

        setShowCall('');
        if (isEndedByMe) {
            socket.emit('endCall', { order: orderChat.id, userId: otherUser.id, type: type });
        }
        
        hideDialog(null);
        
    }

    useEffect(() => {
        if (showCall === 'answer' && !isVideoCall.current) {
            if (currentUserRef.current) {
                currentUserRef.current.srcObject = localStream.current;
            }
        }
    }, [showCall]);

    const CallBtns = () => {
        return (
            <div className={styles.ringingBtns}>
                { showCall === 'answer' ? 
                    <>
                    { hasVideo ? 
                        <VideoCallBtns callInstance={callInstance} currentUserVideoRef={currentUserRef} endCall={endCall} localStream={localStream} /> 
                        : 
                        <AudioCallBtns currentUserVideoRef={currentUserRef} endCall={endCall} /> 
                    }
                    </>
                    :
                    <>
                    { showCall == 'recieving' &&
                        <button onClick={openCall}>
                            {hasVideo ? <i className="ri-vidicon-line"></i> : <i className="ri-phone-fill"></i>}
                        </button>
                    }
                    <button className={styles.endCall} onClick={()=>{hideDialog(null);endCall(true);}}><i className="ri-phone-fill"></i></button>
                    </>
                }
            </div>
        );
    }

    const RingOrRecieve = () => (
        <div className={styles.ringingContent}>
            <span className={`ryb-ringing-img ${styles.ringingImg}`}><UserImage user={otherUser} width={120} height={120} type="user" count={0} /></span>
            { showCall !== 'answer' && <span className={styles.callingText}>{ showCall == 'recieving' ? 'Incoming call' : 'Calling' }</span>}
            <h6 className={styles.ringingName}>{otherUser.fullName}</h6>
            { showCall !== 'answer' && <span className={styles.callType}>{hasVideo ? 'video call' : 'audio call'}</span>}
        </div>
    );

    const callingVidClass = ( showCall == 'ringing' || showCall == undefined ) ? ' ' + styles.callingVidClass : '';

    const ScreenEl = ()=>(
        <>
        <div className={styles.ringingScreen + ' ' + ( hasVideo ? styles.videoCallWrp : styles.audioCallWrp )} style={{ backgroundImage: `url(${otherUser?.userImage})` }}>
            <span className={`minize-btn ${styles.minimize}`} onClick={()=>setMinimized(!minimized)}><i className={minimized ? "ri-arrow-up-s-line" : "ri-arrow-down-s-line"}></i></span>
            <div className={styles.videosWrap + ((showCall == 'answer') ? ' ' + styles.videosWrapCall : '')}>
                { (showCall == 'recieving' || showCall == 'ringing' || showCall == undefined) && <RingOrRecieve />}
                { hasVideo ?
                    <>
                        <Draggable nodeRef={currentUserVidRef} defaultClassName={styles.videoWrp + ' ' + styles.myScreen + callingVidClass}>
                            <div ref={currentUserVidRef}>
                                <video ref={currentUserRef} muted></video>
                            </div>
                        </Draggable>
                        <div className={styles.videoWrp + ' ' + styles.otherScreen}>
                            <video ref={remoteUserRef}></video>
                        </div>
                    </>
                    :
                    <>
                        <audio ref={currentUserRef} muted></audio>
                        <audio ref={remoteUserRef}></audio>
                    </>
                }
            </div>
            { !minimized && <CallBtns /> }
        </div>
        { (showCall == 'ringing' || showCall == undefined) && <audio ref={ringToneRef} src="/sounds/caller.mp3" preload="auto" loop={true}></audio> }
        { showCall == 'recieving' && <audio ref={ringToneRef} src="/sounds/ringtone.mp3" preload="auto" loop={true}></audio> }
        </>
    )
    
    return (
        <>
        { minimized ?
            <Draggable cancel=".minize-btn" nodeRef={nodeRef} bounds={bounds} onStop={handleDragStop} onDrag={handleDrag} position={position}>
                <div ref={nodeRef} className={styles.callScreen + (minimized ? ' ' + styles.minimized : '')}>
                    <ScreenEl />
                </div>
            </Draggable>
            :
            <div className={styles.callScreen + (minimized ? ' ' + styles.minimized : '')}>
                <ScreenEl />
            </div>
        }
        </>
    );
    
};

export default CallScreen;
