import { useEffect, useRef, useState } from 'react';
import WaveSurfer from 'wavesurfer.js';
import styles from "./Bubble.module.css";
import axios from "axios";

function AudioMsgPlayer({audioUrl}) {
  const [isPlaying, setIsPlaying] = useState(false);
  const [duration, setDuration] = useState(0);
  const [playbackRate, setPlaybackRate] = useState(1);
  const waveformRef = useRef(null);
  const wavesurferRef = useRef(null);
  const isInitialized = useRef(false);
  const [currentTime, setCurrentTime] = useState(0);
  const [remainingTime, setRemainingTime] = useState(0);

  useEffect(() => {
    if (!isInitialized.current && audioUrl) {
      isInitialized.current = true;

      axios.get(audioUrl, {
        responseType: 'blob',
      })
      .then((response) => {
        const blobUrl = URL.createObjectURL(response.data);
        wavesurferRef.current = WaveSurfer.create({
            container: waveformRef.current,
            waveColor: '#ced4da',
            progressColor: '#007b5f',
            cursorColor: '#007b5f',
            cursorWidth: 13,
            barWidth: 2,
            barHeight: 15,
            barGap: 2,
            height: 15,
            audioRate: playbackRate,
            // responsive: true,
            normalize: true,
        });

        wavesurferRef.current.load(blobUrl);

        wavesurferRef.current.on('ready', () => {
            setDuration(wavesurferRef.current.getDuration());
            setRemainingTime(Math.floor(wavesurferRef.current.getDuration()));
        });

        wavesurferRef.current.on('seek', (position) => {
            if (wavesurferRef.current) {
                wavesurferRef.current.seekTo(position * wavesurferRef.current.duration());
                setCurrentTime(wavesurferRef.current.getDuration());
            }
        });

        wavesurferRef.current.on('play', () => {
            setIsPlaying(true);
        })

        wavesurferRef.current.on('pause', () => {
            setIsPlaying(false);
        })

        wavesurferRef.current.on('finish', () => {
            setIsPlaying(false);
            setCurrentTime(0);
            wavesurferRef.current.seekTo(0);
        });

        wavesurferRef.current.on('timeupdate', () => {
            const duration = wavesurferRef.current.getDuration();
            const currentTime = wavesurferRef.current.getCurrentTime();
            const remaining = Math.floor(duration - currentTime);
            setRemainingTime(remaining);
        });

        return () => {
            URL.revokeObjectURL(blobUrl);
        };
      })
      .catch((error) => {
        console.error('Error fetching or creating blob URL:', error);
      });

      const updateProgress = () => {
        if (isPlaying) {
          const currentTime = wavesurferRef.current.getCurrentTime();
          const duration = wavesurferRef.current.getDuration();
          const seek = currentTime / duration;
          wavesurferRef.current.seekTo(seek / wavesurferRef.current.getDuration());
          requestAnimationFrame(updateProgress);
        }
      };

      const interval = setInterval(() => {
        if (isPlaying && wavesurferRef.current) {
          updateProgress();
        }
      }, 100);

      return () => {
        if (wavesurferRef.current && duration) {
          wavesurferRef.current.unload();
          wavesurferRef.current = null;
          clearInterval(interval);
        }
      };

    }
  }, []);

  const togglePlay = () => {
    if (wavesurferRef.current.isPlaying()) {
      wavesurferRef.current.pause();
    } else {
      wavesurferRef.current.play();
    }
  };

  const handlePlaybackRateChange = (rate) => {
    setPlaybackRate(rate);
    if (wavesurferRef.current) {
      wavesurferRef.current.setPlaybackRate(rate);
    }
  };

  return (
      <div className={styles.voiceMsg}>
        <button onClick={togglePlay} className={styles.playVoiceIcon}>
          <i className={`ri-${isPlaying ? "pause" : "play"}-fill`}></i>
        </button>
        <div ref={waveformRef} className={`${styles.voiceMsgBar} ryb-msg-bar`}></div>

        <div className={styles.voiceMsgTimes}>
          <span>{formatTime(remainingTime)}</span>
        </div>

        <div className={styles.playbackRate}>
          {playbackRate == 1 && <button onClick={() => handlePlaybackRateChange(1.5)}>1x</button>}
          {playbackRate == 1.5 && <button onClick={() => handlePlaybackRateChange(2)}>1.5x</button>}
          {playbackRate == 2 && <button onClick={() => handlePlaybackRateChange(1)}>2x</button>}
        </div>
      </div>
  );
}

function formatTime(seconds) {
  const minutes = Math.floor(seconds / 60);
  const secs = Math.floor(seconds % 60);
  return `${String(minutes).padStart(2, '0')}:${String(secs).padStart(2, '0')}`;
}

export default AudioMsgPlayer;