import React, { useEffect, useRef } from 'react';
import 'quill/dist/quill.snow.css'; // Import Quill's CSS

const QuillEditor = ({ initialContent, onContentChange }) => {
    const quillRef = useRef(null);
    const editorRef = useRef(null);

    // Initialize Quill editor
    useEffect(() => {
        if (quillRef.current && !editorRef.current) {
            // Dynamically import Quill to avoid SSR issues
            import('quill').then((Quill) => {
                // Create the Quill editor only if it doesn't already exist
                editorRef.current = new Quill.default(quillRef.current, {
                    theme: 'snow',
                    modules: {
                        toolbar: [
                            // [{ header: [1, 2, 3, false] }],
                            ['bold', 'italic', 'underline', 'strike'],
                            [{ list: 'ordered' }, { list: 'bullet' }],
                            ['link', 'image'],
                            ['clean'],
                        ],
                    },
                    placeholder: 'Compose your message...',
                });

                if (initialContent) {
                    editorRef.current.root.innerHTML = initialContent; // Use setHTML to pre-fill the editor
                }

                // Handle text change events
                editorRef.current.on('text-change', () => {
                    const content = editorRef.current.root.innerHTML;
                    onContentChange(content); // Pass the HTML content to the parent component
                });
            });
        }

        // Cleanup Quill editor on unmount
        return () => {
            if (editorRef.current) {
                editorRef.current = null; // Clear the reference
            }
        };
    }, [onContentChange]);

    return (
        <div ref={quillRef} style={{ height: '250px' }}></div>
    );
};

export default QuillEditor;