'use client'
import { useState, useEffect, useCallback, useRef } from 'react'
import { useEditor, EditorContent } from '@tiptap/react'
import Document from '@tiptap/extension-document'
import Dropcursor from '@tiptap/extension-dropcursor'
import StarterKit from '@tiptap/starter-kit'
import TextStyle from '@tiptap/extension-text-style'
import { Color } from '@tiptap/extension-color'
import Underline from '@tiptap/extension-underline'
import TextAlign from '@tiptap/extension-text-align'
import Heading from '@tiptap/extension-heading'
import BulletList from '@tiptap/extension-bullet-list'
import OrderedList from '@tiptap/extension-ordered-list'
import Table from '@tiptap/extension-table'
import TableCell from '@tiptap/extension-table-cell'
import TableHeader from '@tiptap/extension-table-header'
import TableRow from '@tiptap/extension-table-row'
import Image from '@tiptap/extension-image'
import Link from '@tiptap/extension-link'
import WordImporter from './WordImporter'

const PAGE_HEIGHT = 11 * 96 // 11 inches in pixels (96dpi)
const PAGE_WIDTH = 10 * 96 // 8.5 inches
const MARGIN = 1 * 96 // 1 inch margins

export default function WordLikeEditor() {
    const [pages, setPages] = useState([{ id: 1, content: '<p>Start typing...</p>' }])
    const [activePageIndex, setActivePageIndex] = useState(0)
    const editorRef = useRef(null)
    const [linkUrl, setLinkUrl] = useState('')
    const [showLinkInput, setShowLinkInput] = useState(false)
    const [lastSelection, setLastSelection] = useState(null)
    const editor = useEditor({
        extensions: [
            StarterKit,
            Document,
            Dropcursor,
            TextStyle,
            Color,
            Underline,
            TextAlign.configure({ types: ['heading', 'paragraph'] }),
            Heading.configure({ levels: [1, 2, 3] }),
            BulletList,
            OrderedList,
            Table.configure({
                resizable: true,
            }),
            TableRow,
            TableHeader,
            TableCell,
            Image,
            Link.configure({ openOnClick: false }),
        ],
        key: pages[activePageIndex].content,
        content: pages[activePageIndex].content,
        onSelectionUpdate: ({ editor }) => {
            setLastSelection(editor.state.selection)
        },
        onUpdate: ({ editor }) => {
            const updatedPages = [...pages]
            if (lastSelection) {
                setTimeout(() => {
                    editor.commands.setTextSelection(lastSelection, 0)
                }, 0)
            }
            updatedPages[activePageIndex].content = editor.getHTML()
            setPages(updatedPages)
            checkPageOverflow(editor.getHTML(), activePageIndex);
        },
    });

    const handleWordImport = useCallback((htmlContent) => {
        const newPages = splitContentToPages(htmlContent);
        setPages(newPages);
        setActivePageIndex(pages[activePageIndex].id);
    }, []);

    const cleanWordHTML = useCallback((html) => {
        const doc = new DOMParser().parseFromString(html, 'text/html');
        doc.querySelectorAll('o\\:p, v\\:rect, style, xml').forEach(el => el.remove());
        doc.querySelectorAll('span, p').forEach(el => {
            if (el.innerHTML === '&nbsp;' || el.textContent.trim() === '') {
                el.remove();
            } else {
                if (el.tagName.toLowerCase() === 'span') {
                    const parent = el.parentNode;
                    while (el.firstChild) {
                        parent.insertBefore(el.firstChild, el);
                    }
                    parent.removeChild(el);
                }
            }
        });
        return doc.body.innerHTML;
    }, []);

    const handlePaste = useCallback((event) => {
        event.preventDefault();
        const measurementDiv = document.createElement('div');
        if (!editor || !measurementDiv) return;
        const html = event.clipboardData.getData('text/html');
        const text = event.clipboardData.getData('text/plain');
        const cleanedContent = cleanWordHTML(html || `<p>${text}</p>`);
        const newPages = splitContentToPages(cleanedContent);
        setPages(prevPages => {
            const updatedPages = [...prevPages];
            updatedPages.splice(activePageIndex + 1, 0, ...newPages);
            return updatedPages;
        });
        setTimeout(() => {
            setActivePageIndex(activePageIndex + 1);
        }, 10);
    }, [editor, activePageIndex, cleanWordHTML]);

    const splitContentToPages = useCallback((content) => {
        const pages = [];
        const measurementDiv = document.createElement('div');
        let remainingContent = content;
        const measurementEl = document.body.appendChild(measurementDiv);
        try {
            if( remainingContent.length > 0 ){
                while (remainingContent.length > 0) {
                    let low = 0;
                    let high = remainingContent.length;
                    let bestBreak = 0;

                    // Binary search to find optimal break point
                    while (low <= high) {
                        const mid = Math.floor((low + high) / 2);
                        measurementEl.innerHTML = remainingContent.substring(0, mid);
                        const height = measurementEl.offsetHeight;

                        if (height <= PAGE_HEIGHT) {
                            bestBreak = mid;
                            low = mid + 1;
                        } else {
                            high = mid - 1;
                        }
                    }

                    // Find nearest paragraph break
                    let actualBreak = bestBreak;
                    const nextPara = remainingContent.indexOf('<p', bestBreak);
                    const lastParaEnd = remainingContent.lastIndexOf('</p>', bestBreak) + 4;

                    if (nextPara !== -1 && nextPara < bestBreak + 200) {
                        actualBreak = nextPara;
                    } else if (lastParaEnd > 0) {
                        actualBreak = lastParaEnd;
                    }

                    // Ensure we make progress
                    if (actualBreak === 0 && bestBreak > 0) {
                        actualBreak = bestBreak;
                    } else if (actualBreak === 0) {
                        // Force break if no progress
                        actualBreak = Math.min(remainingContent.length, 1000);
                    }

                    const pageContent = remainingContent.substring(0, actualBreak).trim();
                    if (pageContent) {
                        pages.push({
                            id: Date.now() + pages.length,
                            content: pageContent
                        });
                    }
                    remainingContent = remainingContent.substring(actualBreak).trim();
                }
            }
        } finally {
            document.body.removeChild(measurementDiv);
        }

        return pages;
    }, []);

    // Store editor reference
    useEffect(() => {
        editorRef.current = editor
    }, [editor])

    // Check for page overflow
    const checkPageOverflow = useCallback((content, currentPageIndex) => {
        if (!editorRef.current) return;

        // Create measurement container
        const measurementDiv = document.createElement('div');
        measurementDiv.style.position = 'fixed';
        measurementDiv.style.visibility = 'hidden';
        measurementDiv.style.left = '-9999px';
        measurementDiv.style.width = `${PAGE_WIDTH - (2 * MARGIN)}px`;
        measurementDiv.style.padding = `${MARGIN}px`;
        measurementDiv.style.fontFamily = 'Times New Roman';
        measurementDiv.style.fontSize = '12pt';
        measurementDiv.style.lineHeight = '1.5';

        document.body.appendChild(measurementDiv);

        try {
            // Measure total content height
            measurementDiv.innerHTML = content;
            const contentHeight = measurementDiv.offsetHeight;
            const lastCharIsSpace = content.trimEnd().endsWith(' ');
            if (contentHeight > PAGE_HEIGHT - (2 * MARGIN) && !lastCharIsSpace) {
                // Find where content overflows
                let fittingContent = '';
                let overflowContent = '';
                let fittingHeight = 0;

                // Create temporary element to measure content chunks
                const tempContentDiv = document.createElement('div');
                measurementDiv.appendChild(tempContentDiv);

                // Process content line by line
                const lines = content.split('</p>');
                for (let i = 0; i < lines.length; i++) {
                    if (i < lines.length - 1) lines[i] += '</p>'; // Re-add closing tag

                    tempContentDiv.innerHTML = fittingContent + lines[i];
                    const currentHeight = tempContentDiv.offsetHeight;

                    if (currentHeight <= PAGE_HEIGHT - (2 * MARGIN)) {
                        fittingContent += lines[i];
                        fittingHeight = currentHeight;
                    } else {
                        overflowContent += lines[i];
                    }
                }

                measurementDiv.removeChild(tempContentDiv);

                // Update pages if we have overflow
                if (overflowContent) {
                    setPages(prevPages => {
                        const newPages = [...prevPages];

                        // Update current page with fitting content
                        newPages[currentPageIndex].content = fittingContent || '<p></p>';

                        // Insert new page with overflow content
                        newPages.splice(currentPageIndex + 1, 0, {
                            id: Date.now(),
                            content: overflowContent
                        });

                        return newPages;
                    });

                    // Move focus to new page
                    setTimeout(() => setActivePageIndex(currentPageIndex + 1), 0);
                }
            } else if (content === '<p></p>' || content === '<p><br></p>') {
                // Check if we should remove empty page
                setPages(prevPages => {
                    if (prevPages.length <= 1) return prevPages; // Keep at least one page

                    const newPages = [...prevPages];
                    const isLastPage = currentPageIndex === newPages.length - 1;

                    // Remove empty page
                    newPages.splice(currentPageIndex, 1);

                    // Adjust active page index
                    if (isLastPage) {
                        setActivePageIndex(Math.max(0, newPages.length - 1));
                    } else {
                        setActivePageIndex(Math.min(currentPageIndex, newPages.length - 1));
                    }

                    return newPages;
                });
            }
        } finally {
            document.body.removeChild(measurementDiv);
        }
    }, []);

    // Update editor content when active page changes
    useEffect(() => {
        if (!editor || !activePageIndex) return;

        if (editor && pages[activePageIndex]) {
            editor.commands.setContent(pages[activePageIndex].content);
            editor.commands.focus();
            setTimeout(() => editor.commands.focus("end"), 0);
        }
    }, [activePageIndex, editor, pages])

    // Handle link insertion
    const handleAddLink = () => {
        if (linkUrl) {
            editor?.chain().focus().toggleLink({ href: linkUrl }).run()
            setLinkUrl('')
            setShowLinkInput(false)
        }
    }

    const printDocument = (doc) => {
        var printContents = document.getElementById(doc).innerHTML;
        var originalContents = document.body.innerHTML;
        document.body.innerHTML = printContents;
        window.print();
        document.body.innerHTML = originalContents;
    }

    return (
        <section className="ryb-editor-wrap">
            <div className="toolbar">
                {/* Font styling */}
                <div className="toolbar-group">
                    <select
                        value={editor?.getAttributes('heading')?.level || 'paragraph'}
                        onChange={(e) => {
                            const level = e.target.value
                            if (level === 'paragraph') {
                                editor?.chain().focus().setParagraph().run()
                            } else {
                                editor?.chain().focus().toggleHeading({ level: parseInt(level) }).run()
                            }
                        }}
                    >
                        <option value="paragraph">Normal</option>
                        <option value="1">Heading 1</option>
                        <option value="2">Heading 2</option>
                        <option value="3">Heading 3</option>
                    </select>

                    <select
                        value={editor?.getAttributes('textStyle')?.fontFamily || 'Times New Roman'}
                        onChange={(e) => editor?.chain().focus().setFontFamily(e.target.value).run()}
                    >
                        <option value="Times New Roman">Times New Roman</option>
                        <option value="Arial">Arial</option>
                        <option value="Courier New">Courier New</option>
                        <option value="Georgia">Georgia</option>
                    </select>

                    <select
                        value={editor?.getAttributes('textStyle')?.fontSize || '12pt'}
                        onChange={(e) => editor?.chain().focus().setFontSize(e.target.value).run()}
                    >
                        <option value="8pt">8pt</option>
                        <option value="10pt">10pt</option>
                        <option value="12pt">12pt</option>
                        <option value="14pt">14pt</option>
                        <option value="18pt">18pt</option>
                        <option value="24pt">24pt</option>
                    </select>
                </div>

                {/* Text formatting */}
                <div className="toolbar-group">
                    <button
                        onClick={() => editor?.chain().focus().toggleBold().run()}
                        className={editor?.isActive('bold') ? 'is-active' : ''}
                        title="Bold"
                    >
                        <i className="ri-bold"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().toggleItalic().run()}
                        className={editor?.isActive('italic') ? 'is-active' : ''}
                        title="Italic"
                    >
                        <i className="ri-italic"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().toggleUnderline().run()}
                        className={editor?.isActive('underline') ? 'is-active' : ''}
                        title="Underline"
                    >
                        <i className="ri-underline"></i>
                    </button>
                    <button
                        onClick={() => editor.chain().focus().setColor('#958DF1').run()}
                        className={editor?.isActive('textStyle', { color: '#000000' }) ? 'is-active' : ''}
                        title="Text Color"
                    >
                        <i className="ri-font-color"></i>
                    </button>
                </div>

                {/* Paragraph formatting */}
                <div className="toolbar-group">
                    <button
                        onClick={() => editor?.chain().focus().setTextAlign('left').run()}
                        className={editor?.isActive({textAlign: 'left'}) ? 'is-active' : ''}
                        title="Align Left"
                    >
                        <i className="ri-align-left"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().setTextAlign('center').run()}
                        className={editor?.isActive({textAlign: 'center'}) ? 'is-active' : ''}
                        title="Align Center"
                    >
                        <i className="ri-align-center"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().setTextAlign('right').run()}
                        className={editor?.isActive({textAlign: 'right'}) ? 'is-active' : ''}
                        title="Align Right"
                    >
                        <i className="ri-align-right"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().setTextAlign('justify').run()}
                        className={editor?.isActive({textAlign: 'justify'}) ? 'is-active' : ''}
                        title="Justify"
                    >
                        <i className="ri-align-justify"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().toggleBulletList().run()}
                        className={editor?.isActive('bulletList') ? 'is-active' : ''}
                        title="Bullet List"
                    >
                        <i className="ri-list-unordered"></i>
                    </button>
                    <button
                        onClick={() => editor?.chain().focus().toggleOrderedList().run()}
                        className={editor?.isActive('orderedList') ? 'is-active' : ''}
                        title="Numbered List"
                    >
                        <i className="ri-list-ordered"></i>
                    </button>
                </div>

                {/* Links */}
                <div className="toolbar-group">
                    {showLinkInput ? (
                        <div className="link-input">
                            <input
                                type="text"
                                placeholder="Enter URL"
                                value={linkUrl}
                                onChange={(e) => setLinkUrl(e.target.value)}
                                onKeyDown={(e) => e.key === 'Enter' && handleAddLink()}
                            />
                            <button onClick={handleAddLink}>Apply</button>
                            <button onClick={() => setShowLinkInput(false)}>Cancel</button>
                        </div>
                    ) : (
                        <button
                            onClick={() => {
                                if (editor?.isActive('link')) {
                                    editor?.chain().focus().unsetLink().run()
                                } else {
                                    setShowLinkInput(true)
                                }
                            }}
                            className={editor?.isActive('link') ? 'is-active' : ''}
                            title="Link"
                        >
                            <i className="ri-link"></i>
                        </button>
                    )}
                </div>
            </div>
            <WordImporter onImport={handleWordImport} />
            <div className="editor-container">
                <div className="editor-side">
                    <div className="editor-side-header">
                        <div className="editor-side-header-title">
                            <h5>Document</h5>
                        </div>
                        <ul className={`editor-side-header-actions ${pages.length > 1 ? 'has-actions' : ''}`}>
                            <li><i className="ri-file-2-line"></i><b>Name:</b>sfsfsdfsfret.docx</li>
                            <li><i className="ri-cloud-line"></i><b>Size:</b>785 KB</li>
                            <li><i className="ri-time-line"></i><b>Date Created:</b>12/12/2021</li>
                            <li><i className="ri-time-fill"></i><b>Last Modified:</b>25/4/2025</li>
                            <li><i className="ri-user-2-line"></i><b>Creator:</b>Mohamed Montaser</li>
                            <li><i className="ri-download-2-line"></i>Download this document</li>
                            <li onClick={()=> printDocument("pages")} style={{cursor: "pointer"}}><i className="ri-printer-line"></i>Print this document</li>
                        </ul>
                    </div>
                </div>
                <div className="pages" id="pages">
                    {pages.map((page, index) => (
                        <div
                            key={page.id}
                            className={`page ${index === activePageIndex ? 'active' : ''}`}
                            style={{
                                width: `${PAGE_WIDTH}px`,
                                minHeight: `${PAGE_HEIGHT}px`,
                                margin: '0 auto 1rem',
                            }}
                        >
                            {index === activePageIndex ? (
                                <EditorContent
                                    editor={editor}
                                    onPaste={(e) => handlePaste(e, editor)}
                                    style={{
                                        padding: `${MARGIN}px`,
                                        minHeight: `calc(100% - ${2 * MARGIN}px)`,
                                        outline: 'none'
                                    }}
                                />
                            ) : (
                                <div
                                    className="page-preview"
                                    style={{
                                        padding: `${MARGIN}px`,
                                        minHeight: `calc(100% - ${2 * MARGIN}px)`,
                                    }}
                                    dangerouslySetInnerHTML={{ __html: page.content }}
                                    onClick={() => setActivePageIndex(index)}
                                />
                            )}
                            <div className="page-number">Page {index + 1}</div>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    )
}