import { useState, useCallback, useRef } from 'react';
import axios, { CancelTokenSource } from 'axios';

type Message = {
    role: 'user' | 'assistant' | 'system';
    content: string;
};

export const useQwenStream = () => {
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [generatedText, setGeneratedText] = useState('');
    const [conversation, setConversation] = useState<Message[]>([]);
    const responseBuffer = useRef('');
    const cancelTokenSource = useRef<CancelTokenSource | null>(null);
    const reset = useCallback(() => {
        setGeneratedText('');
        setError(null);
        setConversation([]);
        responseBuffer.current = '';
    }, []);
    const stopGeneration = useCallback(() => {
        if (cancelTokenSource.current) {
            cancelTokenSource.current.cancel('Generation stopped by user');
            cancelTokenSource.current = null;
            setIsLoading(false);
        }
    }, []);
    const streamQwenResponse = useCallback(async (newMessage: string) => {
        setIsLoading(true);
        setError(null);
        setGeneratedText('');
        responseBuffer.current = '';
        cancelTokenSource.current = axios.CancelToken.source();

        let text = '';
        setConversation(prev => [
            ...prev,
            { role: 'user', content: newMessage },
        ]);

        const updatedMessages = [
            ...conversation,
            { role: 'user' as const, content: newMessage }
        ];

        try {
            const response = await axios({
                method: 'POST',
                url: 'https://dashscope-intl.aliyuncs.com/compatible-mode/v1/chat/completions',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${import.meta.env.VITE_QWEN_API_KEY}`,
                },
                data: {
                    model: 'qwen2.5-72b-instruct',
                    messages: updatedMessages,
                    stream:true,
                    parameters: {
                        incremental_output: true
                    },
                    cancelToken: cancelTokenSource.current.token,
                },
                onDownloadProgress: (progressEvent) => {
                    try {
                        const chunk = progressEvent.event.target.responseText;
                        const newData = chunk.slice(responseBuffer.current.length);
                        responseBuffer.current = chunk;
                        const lines = newData.split('\n').filter((str:any) => str.trim() !== '');
                        for (const line of lines) {
                            const trimmedLine = line.trim();
                            if (!trimmedLine || trimmedLine.includes('[DONE]')) continue;
                            if (trimmedLine.startsWith('data: ')) {
                                const jsonPart = trimmedLine.slice(6); // Remove "data: " prefix
                                const jsonStr = jsonPart.trim();
                                if (!jsonStr) continue;
                                try {
                                    const data = JSON.parse(jsonStr);
                                    const content = data.choices?.[0]?.message?.content ||
                                        data.message?.content ||
                                        data.choices?.[0]?.delta?.content ||
                                        data.delta?.content ||
                                        data.content;

                                    if (content) {
                                        text += content;
                                    }

                                } catch (e) {
                                    console.error('Error parsing JSON:', jsonStr, e);
                                }
                            }
                        }
                    } catch (e) {
                        console.error('Error parsing chunk:', e);
                    }
                }
            });
            setGeneratedText(text);
        } catch (err) {
            console.error('Error:', err);
            let errorMessage = 'Failed to get response from Qwen';

            if (axios.isAxiosError(err)) {
                if (err.response?.data?.message?.includes('missing_required_parameter')) {
                    errorMessage = 'Invalid request format. Please check your messages parameter.';
                } else if (err.response?.status === 401) {
                    errorMessage = 'Invalid API key. Please check your credentials.';
                } else {
                    errorMessage = err.response?.data?.message || err.message;
                }
            }

            setError(errorMessage);
        } finally {
            setIsLoading(false);
        }
    }, [conversation, generatedText]);

    return {
        generatedText,
        isLoading,
        error,
        setConversation,
        setGeneratedText,
        conversation,
        streamQwenResponse,
        stopGeneration,
        reset
    };
};
